import time
from threading import Thread

import atexit
import bpy
import random
from mathutils import Vector, Matrix, Quaternion, Euler
from ..rebocap_api import *
from typing import Optional
from .utils import show_message_box
from ..rebocap_api.rebocap_ws_sdk import REBOCAP_JOINT_NAMES


class RebocapWsSdkV1(RebocapWsSdk):

    def __init__(self, coordinate_type: CoordinateType = CoordinateType.DefaultCoordinate, use_global_rotation=False):
        super().__init__(coordinate_type, use_global_rotation)
        super().set_pose_msg_callback(self.pose_msg_callback_own)
        self.last_msg = None
        self.record_list = None

    def pose_msg_callback_own(self, parent: RebocapWsSdk, trans: list, pose24: list, static_index: int, tp: float):
        self.last_msg = (trans, pose24, static_index, tp)
        if self.record_list is not None:
            self.record_list.append(self.last_msg)

    def get_last_msg(self):
        return self.last_msg

    def start_record(self):
        self.record_list = []

    def stop_record(self):
        record_list = self.record_list
        self.record_list = None
        return record_list


rebocap_app: Optional[RebocapWsSdkV1] = None

rebocap_timer = None
rebocap_connected = False
drive_direct_when_connect = True


class FrameData:
    def __init__(self, trans, pose24, static_index, ts) -> None:
        self.trans = trans
        self.pose24 = pose24
        self.static_index = static_index
        self.ts = ts


joints = [
    "mixamorig:Hips",
    "mixamorig:LeftUpLeg",
    "mixamorig:RightUpLeg",
    "mixamorig:Spine",
    "mixamorig:LeftLeg",
    "mixamorig:RightLeg",
    "mixamorig:Spine1",
    "mixamorig:LeftFoot",
    "mixamorig:RightFoot",
    "mixamorig:Spine2",
    "mixamorig:LeftToeBase",
    "mixamorig:RightToeBase",
    "mixamorig:Neck",
    "mixamorig:LeftShoulder",
    "mixamorig:RightShoulder",
    "mixamorig:Head",
    "mixamorig:LeftArm",
    "mixamorig:RightArm",
    "mixamorig:LeftForeArm",
    "mixamorig:RightForeArm",
    "mixamorig:LeftHand",
    "mixamorig:RightHand",
    "mixamorig:LeftHandIndex1",
    "mixamorig:RightHandIndex1"
]

idx_vec = [name.lower() for name in REBOCAP_JOINT_NAMES]

stop_debug_thread = False

class BoneInfo:
    def __init__(self) -> None:
        self.rest_matrix_to_world = None
        self.rest_matrix_from_world = None
    
    def is_rest_init(self):
        if self.rest_matrix_to_world is None or self.rest_matrix_from_world is None:
            return False
        return True

    def rest_init(self, rest_matrix_to_world, rest_matrix_from_world):
        self.rest_matrix_to_world = rest_matrix_to_world
        self.rest_matrix_from_world = rest_matrix_from_world

    def get_rest_init(self):
        return (self.rest_matrix_to_world, self.rest_matrix_from_world)

class ObjInfo:
    def __init__(self) -> None:
        self.bone_map = {}
        self.root_bone = None
        self.matrix_global_hip = None
        self.matrix_global_2_root_inverse = None
        self.matrix_root_2_hip_inverse = None

    def get_bone(self, name):
        if name in self.bone_map:
            return self.bone_map[name]
        return None

    def insert_bone(self, name):
        bone = BoneInfo()
        self.bone_map[name] = bone
        return bone

    def get_or_insert_bone(self, name):
        bone = self.get_bone(name)
        if bone is None:
            bone = self.insert_bone(name)
        return bone

    def set_root_info(self, root_bone, matrix_global_hip, matrix_global_2_root_inverse, matrix_root_2_hip_inverse):
        self.root_bone = root_bone
        self.matrix_global_hip = matrix_global_hip
        self.matrix_global_2_root_inverse = matrix_global_2_root_inverse
        self.matrix_root_2_hip_inverse = matrix_root_2_hip_inverse


class ObjInfoHelper:
    def __init__(self) -> None:
        self.obj_map = {}

    def get_obj(self, name):
        if name in self.obj_map:
            return self.obj_map[name]
        return None

    def insert_obj(self, name):
        obj = ObjInfo()
        self.obj_map[name] = obj
        return obj

    def get_or_insert_obj(self, name):
        obj = self.get_obj(name)
        if obj is None:
            obj = self.insert_obj(name)
        return obj


obj_info_helper = ObjInfoHelper()


def t_run():
    global stop_debug_thread
    print(f'started!!!!!!!')
    while True:
        time.sleep(1.0)
        print(f'obj found')
        continue
        for obj in bpy.context.scene.objects:
            if obj.type != 'ARMATURE':
                continue
            continue
        pass


# 注册退出函数
if sys.gettrace() is not None:
    t = Thread(target=t_run)
    t.start()


def init_rebocap_api():
    global rebocap_app
    print(f'first init rebocap api')
    rebocap_app = RebocapWsSdkV1(coordinate_type=CoordinateType.BlenderCoordinate)
    print(f'finish init rebocap api')


def uninit_rebocap_api():
    global rebocap_app
    if rebocap_app is not None:
        rebocap_app.close()
        # rebocap_app.__del__()
    rebocap_app = None


def show_error_message(self, ctx):
    self.layout.label(text=ctx.scene.error_msg)


def get_pose_idx(bone: bpy.types.PoseBone):
    if bone.rebocap_init:
        return bone.rebocap_pose_idx
    name_l = bone.name.lower()
    for idx, name in enumerate(idx_vec):
        if name.lower() == name_l:
            bone.rebocap_pose_idx = idx
            break
    bone.rebocap_init = True
    return bone.rebocap_pose_idx


def show_error(msg, ctx):
    ctx.scene.error_msg = msg
    bpy.context.window_manager.popup_menu(
        show_error_message, title="Error", icon='ERROR')


class CustomObj:
    def __init__(self, obj: bpy.types.Object) -> None:
        self.obj = None
        self.vec = []
        self.update_obj(obj)

    def update(self, obj: bpy.types.Object):
        if self.obj == obj:
            return
        self.update_obj(obj)

    def update_obj(self, obj: bpy.types.Object):
        self.vec = [None for _ in range(24)]
        for bone in obj.pose.bones:
            name = bone.name.lower()
            for idx, item in enumerate(idx_vec):
                if item.lower() in name:
                    self.vec[idx] = (bone, obj.data.bones.get(bone.name))
                    break
        self.obj = obj


class RebocapConnect(bpy.types.Operator):
    bl_idname = 'rebocap.connect'
    bl_label = 'Connect'
    pelvis_offset = [0.0, 0.0, 0.0]

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.reinit_data()
        self.root_bone_matrix_inverse = None

    def reinit_data(self):
        self.obj_map = {}
        self.root_bone = None
        global obj_info_helper
        obj_info_helper = ObjInfoHelper()
        self.root_relative_position = Vector((0.0, 0.0, 0.0))
        self.root_bone_matrix_inverse = None

    def execute(self, ctx):
        global rebocap_app, rebocap_connected, drive_direct_when_connect

        source_obj = None
        for obj in bpy.context.scene.objects:
            if obj.type != 'ARMATURE':
                continue

            if not hasattr(obj, 'rebocap_source_armature'):
                continue

            is_direct = obj.rebocap_drive_type == 'DIRECT'
            if is_direct:
                drive_direct_when_connect = True
                if obj.pose.bones[0].name.startswith('mixamorig'):
                    source_obj = obj
            else:
                drive_direct_when_connect = False
                source_obj = bpy.data.objects.get(obj.rebocap_source_armature)
            if source_obj is None:
                continue

        if source_obj is None:
            show_message_box(message='rebocap source is not set, please set source avatar!!!')
            return {'no source found'}

        rebocap_bone_map = ctx.scene.rebocap_bone_map
        avatar = source_obj
        down_body = [
            0, 1, 4, 7, 2, 5, 8  # root, left legs,  right legs
        ]

        bones = [avatar.pose.bones.get(getattr(rebocap_bone_map, f'node_{e}', '')) for e in down_body]
        if '' in bones:
            show_message_box(message='down body not all bind, please set target bone for Legs and Pelvis')
            return {''}

        res = rebocap_app.open(ctx.scene.port, uid=random.randint(1, 100000))
        if res != 0:
            show_error(f"Connect Fail. ret={res}", ctx)
            return {'CANCELLED'}

        self.reinit_data()
        ctx.window_manager.modal_handler_add(self)
        global rebocap_timer
        rebocap_timer = ctx.window_manager.event_timer_add(1 / 60, window=ctx.window)
        ctx.scene.open = True
        rebocap_connected = True
        return {'RUNNING_MODAL'}

    def drive_direct(self, trans, pose24, static_index, ts):
        for obj in bpy.context.scene.objects:
            if obj.type != 'ARMATURE' or obj.rebocap_drive_type != 'DIRECT':
                continue

            custom_obj = None
            global obj_info_helper
            obj_info = obj_info_helper.get_or_insert_obj(obj.name)

            rebocap_bone_map = bpy.context.scene.rebocap_bone_map
            if not hasattr(obj, 'rebocap_source_armature'):
                continue
            source_obj = bpy.data.objects.get(obj.rebocap_source_armature)
            if source_obj is None or source_obj != obj:
                continue

            if obj.name not in self.obj_map:
                custom_obj = CustomObj(obj)
                self.obj_map[obj.name] = custom_obj
            else:
                custom_obj = self.obj_map[obj.name]
                custom_obj.update(obj)
            for pose_bone in obj.pose.bones:
                idx = get_pose_idx(pose_bone)
                if idx == -1 or idx in (10, 11, 22, 23):
                    continue

                bone_name = pose_bone.name
                bone_info = obj_info.get_or_insert_bone(bone_name)
                if not bone_info.is_rest_init():
                    data_bone = obj.data.bones.get(bone_name)
                    rest_matrix_to_world = obj.matrix_world.to_quaternion() @ data_bone.matrix_local.to_quaternion()
                    rest_matrix_from_world = rest_matrix_to_world.inverted()
                    bone_info.rest_init(rest_matrix_to_world, rest_matrix_from_world)
                else:
                    rest_matrix_to_world, rest_matrix_from_world = bone_info.get_rest_init()

                pose = pose24[idx]
                if idx == 0:
                    if obj_info.root_bone is None:
                        current_hip_bone = pose_bone
                        root_bone = pose_bone
                        while root_bone.parent is not None:
                            root_bone = root_bone.parent

                        matrix_root_2_hip = obj.data.bones[root_bone.name].matrix_local.inverted() @ obj.data.bones[current_hip_bone.name].matrix_local
                        matrix_global_2_root = obj.matrix_world @ obj.data.bones.get(root_bone.name).matrix_local
                        matrix_global_hip = obj.matrix_world @ obj.data.bones[current_hip_bone.name].matrix_local
                        obj_info.set_root_info(root_bone, matrix_global_hip, matrix_global_2_root.inverted(), matrix_root_2_hip.inverted())

                    origin_global_hip_trans = Vector((trans[0], trans[1], trans[2]))
                    new_global_hip: Matrix = obj_info.matrix_global_hip
                    new_global_hip.translation = origin_global_hip_trans

                    new_local_matrix = obj_info.matrix_global_2_root_inverse @ new_global_hip @ obj_info.matrix_root_2_hip_inverse
                    obj_info.root_bone.location = new_local_matrix.translation

                new_pose = rest_matrix_from_world @ Quaternion([pose[3], pose[0], pose[1], pose[2]]) @ rest_matrix_to_world
                pose_bone.rotation_quaternion = new_pose

    def drive_retarget(self, trans, pose24, static_index, ts):
        for obj in bpy.context.scene.objects:
            if obj.type != 'ARMATURE' or obj.rebocap_drive_type != 'RETARGET':
                continue

            rebocap_bone_map = bpy.context.scene.rebocap_bone_map
            if not hasattr(obj, 'rebocap_source_armature'):
                continue
            source_obj = bpy.data.objects.get(obj.rebocap_source_armature)
            if source_obj is None or source_obj != obj:
                continue

            map_bones_names = [
                getattr(rebocap_bone_map, f'node_{i}') for i in range(22)
            ]
            map_bones = [
                source_obj.pose.bones.get(map_bones_names[i]) if map_bones_names[i] != '' else None for i in range(22)
            ]

            global obj_info_helper
            obj_info = obj_info_helper.get_or_insert_obj(obj.name)

            # handle merge pose for spine and chest
            hip_bone = map_bones[0]
            spine_bone = map_bones[3]
            chest_bone = map_bones[6]
            up_chest_bone = map_bones[9]
            left_shoulder_bone = map_bones[13]
            right_shoulder_bone = map_bones[14]
            left_arm_bone = map_bones[16]
            right_arm_bone = map_bones[17]
            all_pose = []
            for pose in pose24:
                all_pose.append(Quaternion([pose[3], pose[0], pose[1], pose[2]]))

            if spine_bone is not None or chest_bone is not None or up_chest_bone is not None:
                if spine_bone is None:
                    all_pose[6] = all_pose[3] @ all_pose[6]
                if chest_bone is None and up_chest_bone is None:
                    all_pose[3] = all_pose[3] @ all_pose[6] @ all_pose[9]
                elif chest_bone is None:
                    all_pose[9] = all_pose[6] @ all_pose[9]
                elif up_chest_bone is None:
                    all_pose[6] = all_pose[6] @ all_pose[9]
            if left_shoulder_bone is None and left_arm_bone is not None:
                all_pose[16] = all_pose[13] @ all_pose[16]
            if right_shoulder_bone is None and right_arm_bone is not None:
                all_pose[17] = all_pose[14] @ all_pose[17]

            # print(all_pose[1])
            for idx, bone in enumerate(map_bones):
                if bone is None or idx == -1 or idx in (10, 11, 22, 23):
                    if idx == 0:  # hip must bind!!!
                        break
                    continue
                bone_name = bone.name
                bone_info = obj_info.get_or_insert_bone(bone_name)
                if not bone_info.is_rest_init():
                    data_bone = source_obj.data.bones.get(map_bones_names[idx])
                    bone.rotation_mode = 'QUATERNION'
                    rest_matrix_to_world = obj.matrix_world.to_quaternion() @ data_bone.matrix_local.to_quaternion()
                    rest_matrix_from_world = rest_matrix_to_world.inverted()
                    bone_info.rest_init(rest_matrix_to_world, rest_matrix_from_world)
                else:
                    rest_matrix_to_world, rest_matrix_from_world = bone_info.get_rest_init()

                if idx == 0:
                    if obj_info.root_bone is None:
                        current_hip_bone = bone
                        root_bone = bone
                        while root_bone.parent is not None:
                            root_bone = root_bone.parent

                        matrix_root_2_hip = obj.data.bones[root_bone.name].matrix_local.inverted() @ obj.data.bones[current_hip_bone.name].matrix_local
                        matrix_global_2_root = obj.matrix_world @ obj.data.bones.get(root_bone.name).matrix_local
                        matrix_global_hip = obj.matrix_world @ obj.data.bones[current_hip_bone.name].matrix_local
                        obj_info.set_root_info(root_bone, matrix_global_hip, matrix_global_2_root.inverted(), matrix_root_2_hip.inverted())

                    origin_global_hip_trans = Vector((trans[0], trans[1], trans[2]))
                    new_global_hip: Matrix = obj_info.matrix_global_hip
                    new_global_hip.translation = origin_global_hip_trans

                    new_local_matrix = obj_info.matrix_global_2_root_inverse @ new_global_hip @ obj_info.matrix_root_2_hip_inverse
                    obj_info.root_bone.location = new_local_matrix.translation

                pose = all_pose[idx]
                new_pose = rest_matrix_from_world @ pose @ rest_matrix_to_world
                bone.rotation_quaternion = new_pose

    def modal(self, ctx, evt):
        global rebocap_app, rebocap_connected, drive_direct_when_connect
        if not rebocap_connected:
            return {'PASS_THROUGH'}

        last_data = rebocap_app.get_last_msg()
        if last_data is not None:
            trans, pose24, static_index, ts = last_data

            if drive_direct_when_connect:
                self.drive_direct(trans, pose24, static_index, ts)
            else:
                self.drive_retarget(trans, pose24, static_index, ts)
        return {'PASS_THROUGH'}


def stop_record_retarget(ctx, record_list, obj):
    begin_t = record_list[0][3]
    rebocap_bone_map = bpy.context.scene.rebocap_bone_map
    source_obj = bpy.data.objects.get(obj.rebocap_source_armature)
    map_bones_names = [
        getattr(rebocap_bone_map, f'node_{i}') for i in range(22)
    ]
    map_bones = [
        source_obj.pose.bones.get(map_bones_names[i]) if map_bones_names[i] != '' else None for i in range(22)
    ]

    if source_obj is None:
        return
    all_pose_list = []
    for record in record_list:
        all_pose_list.append([Quaternion([pose[3], pose[0], pose[1], pose[2]]) for pose in record[1]])
    hip_bone = map_bones[0]
    spine_bone = map_bones[3]
    chest_bone = map_bones[6]
    up_chest_bone = map_bones[9]
    left_shoulder_bone = map_bones[13]
    right_shoulder_bone = map_bones[14]
    left_arm_bone = map_bones[16]
    right_arm_bone = map_bones[17]
    if spine_bone is not None or chest_bone is not None or up_chest_bone is not None:
        if spine_bone is None:
            for all_pose in all_pose_list:
                all_pose[6] = all_pose[3] @ all_pose[6]
        if chest_bone is None and up_chest_bone is None:
            for all_pose in all_pose_list:
                all_pose[3] = all_pose[3] @ all_pose[6] @ all_pose[9]
        elif chest_bone is None:
            for all_pose in all_pose_list:
                all_pose[9] = all_pose[6] @ all_pose[9]
        elif up_chest_bone is None:
            for all_pose in all_pose_list:
                all_pose[6] = all_pose[6] @ all_pose[9]
    if left_shoulder_bone is None and left_arm_bone is not None:
        for all_pose in all_pose_list:
            all_pose[16] = all_pose[13] @ all_pose[16]
    if right_shoulder_bone is None and right_arm_bone is not None:
        for all_pose in all_pose_list:
            all_pose[17] = all_pose[14] @ all_pose[17]

    obj.animation_data_create()
    action = bpy.data.actions.new(name='rebocap')
    obj.animation_data.action = action
    dt = 1 / ctx.scene.render.fps
    global obj_info_helper
    obj_info = obj_info_helper.get_or_insert_obj(obj.name)

    for idx, bone in enumerate(map_bones):
        if bone is None or idx == -1 or idx in (10, 11, 22, 23):
            if idx == 0:  # hip must bind!!!
                break
            continue
        bone_name = bone.name
        bone_info = obj_info.get_or_insert_bone(bone_name)
        rest_matrix_to_world, rest_matrix_from_world = bone_info.get_rest_init()
        rotation_quaternion = []
        t = []
        for fn, frame in enumerate(record_list):
            t.append((frame[3] - begin_t) / dt + 1)
            pose = all_pose_list[fn][idx]
            new_pose = rest_matrix_from_world @ pose @ rest_matrix_to_world
            rotation_quaternion.append(new_pose)
        data_path = 'pose.bones["%s"].rotation_quaternion' % bone.name
        for axis_i in range(4):
            curve = action.fcurves.new(data_path=data_path, index=axis_i)
            keyframe_points = curve.keyframe_points
            frame_count = len(rotation_quaternion)
            keyframe_points.add(frame_count)
            for i, rotation_quaternion_item in enumerate(rotation_quaternion):
                keyframe_points[i].co = (
                    t[i],
                    rotation_quaternion_item[axis_i]
                )

        if idx == 0 and obj_info.root_bone is not None:
            bone = obj_info.root_bone
            data_path = 'pose.bones["%s"].location' % bone.name

            all_trans_new = []
            for i, frame in enumerate(record_list):
                trans = frame[0]
                origin_global_hip_trans = Vector((trans[0], trans[1], trans[2]))
                new_global_hip: Matrix = obj_info.matrix_global_hip
                new_global_hip.translation = origin_global_hip_trans

                new_local_matrix = obj_info.matrix_global_2_root_inverse @ new_global_hip @ obj_info.matrix_root_2_hip_inverse
                all_trans_new.append(new_local_matrix.translation)

            for axis_i in range(3):
                curve = action.fcurves.new(data_path=data_path, index=axis_i)
                keyframe_points = curve.keyframe_points
                frame_count = len(record_list)
                keyframe_points.add(frame_count)

                for i, frame in enumerate(record_list):
                    keyframe_points[i].co = (
                        t[i],
                        all_trans_new[i][axis_i]
                    )


def stop_record(ctx):
    if ctx.scene.recording is False:
        return
    ctx.scene.recording = False
    global rebocap_app
    record_list = rebocap_app.stop_record()
    begin_t = record_list[0][3]
    action = bpy.data.actions.new(name='rebocap')
    for obj in bpy.context.scene.objects:
        if obj.type != 'ARMATURE':
            continue
        if not hasattr(obj, 'rebocap_source_armature'):
            continue
        source_obj = bpy.data.objects.get(obj.rebocap_source_armature)
        if source_obj is None or source_obj != obj:
            continue

        is_direct = obj.rebocap_drive_type == 'DIRECT'
        is_valid = False
        if is_direct:
            for pose_bone in obj.pose.bones:
                if is_direct:
                    idx = get_pose_idx(pose_bone)
                    if idx != -1:
                        is_valid = True
                        break
            if not is_valid:
                continue
        obj.animation_data_create()
        obj.animation_data.action = action
        dt = 1 / ctx.scene.render.fps
        if not is_direct:
            stop_record_retarget(ctx, record_list, obj)
            break

        global obj_info_helper
        obj_info = obj_info_helper.get_or_insert_obj(obj.name)
        # below is direct
        for pose_bone in obj.pose.bones:
            bone = obj.data.bones.get(pose_bone.name)
            idx = get_pose_idx(pose_bone)
            if idx == -1 or idx in (10, 11, 22, 23):
                continue

            bone_info = obj_info.get_or_insert_bone(bone.name)
            rest_matrix_to_world, rest_matrix_from_world = bone_info.get_rest_init()
            rotation_quaternion = []
            t = []
            for frame in record_list:
                t.append((frame[3] - begin_t) / dt + 1)
                pose = frame[1][idx]
                new_pose = rest_matrix_from_world @ Quaternion([pose[3], pose[0], pose[1], pose[2]]) @ rest_matrix_to_world
                rotation_quaternion.append(new_pose)
            data_path = 'pose.bones["%s"].rotation_quaternion' % pose_bone.name
            for axis_i in range(4):
                curve = action.fcurves.new(data_path=data_path, index=axis_i)
                keyframe_points = curve.keyframe_points
                frame_count = len(rotation_quaternion)
                keyframe_points.add(frame_count)
                for i, rotation_quaternion_item in enumerate(rotation_quaternion):
                    keyframe_points[i].co = (
                        t[i],
                        rotation_quaternion_item[axis_i]
                    )
            if idx == 0 and obj_info.root_bone is not None:
                data_path = 'pose.bones["%s"].location' % pose_bone.name
                all_trans_new = []
                for i, frame in enumerate(record_list):
                    trans = frame[0]
                    origin_global_hip_trans = Vector((trans[0], trans[1], trans[2]))
                    new_global_hip: Matrix = obj_info.matrix_global_hip
                    new_global_hip.translation = origin_global_hip_trans

                    new_local_matrix = obj_info.matrix_global_2_root_inverse @ new_global_hip @ obj_info.matrix_root_2_hip_inverse
                    all_trans_new.append(new_local_matrix.translation)

                for axis_i in range(3):
                    curve = action.fcurves.new(data_path=data_path, index=axis_i)
                    keyframe_points = curve.keyframe_points
                    frame_count = len(record_list)
                    keyframe_points.add(frame_count)

                    for i, frame in enumerate(record_list):
                        keyframe_points[i].co = (
                            t[i],
                            all_trans_new[i][axis_i]
                        )
    for cu in action.fcurves:
        for bez in cu.keyframe_points:
            bez.interpolation = 'LINEAR'


class RebocapDisconnect(bpy.types.Operator):
    bl_idname = 'rebocap.disconnect'
    bl_label = 'Disconnect'

    def execute(self, ctx):
        stop_record(ctx)
        ctx.scene.open = False
        global rebocap_app, rebocap_connected
        rebocap_app.close()
        global rebocap_timer
        ctx.window_manager.event_timer_remove(rebocap_timer)
        rebocap_connected = False
        global obj_info_helper
        obj_info_helper = ObjInfoHelper()
        return {'FINISHED'}


class RebocapStartRecord(bpy.types.Operator):
    bl_idname = 'rebocap.start_record'
    bl_label = 'Start Record'

    def execute(self, ctx):
        global rebocap_app
        rebocap_app.start_record()
        ctx.scene.recording = True
        return {'FINISHED'}


class RebocapStopRecord(bpy.types.Operator):
    bl_idname = 'rebocap.stop_record'
    bl_label = 'Stop Record'

    def execute(self, ctx):
        stop_record(ctx)
        return {'FINISHED'}
